"""Mark utilities for Simulation C.

This module provides helper functions to generate mark rows and to
compute the fraction of histories that remain unmarked given a mark
probability and number of opportunities.  In the simplified analytic
driver these functions are not invoked directly but are included for
completeness and to guide a future full walker implementation.
"""

from __future__ import annotations

from typing import List


def mark_rows(y_slit: int, N: int, row_stride: int = 3) -> List[int]:
    """Generate N row indices after the slits for mark opportunities.

    The first mark row is ``y_slit + 3``; subsequent rows are spaced
    by ``row_stride``.  Returns a list of row indices of length N.
    """
    rows = []
    y = y_slit + 3
    for _ in range(N):
        rows.append(y)
        y += row_stride
    return rows


def fraction_unmarked(p: float, N: int) -> float:
    """Return the expected fraction of histories that remain unmarked.

    Each of the N independent opportunities has probability p of
    producing a mark.  The probability of no mark over all N trials is
    (1-p)^N.
    """
    if p < 0:
        p = 0.0
    if p > 1:
        p = 1.0
    return (1.0 - p) ** N